/* benchmark.cc
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * This program runs benchmark comparisons of the implemented algorithms.
 */

#include <iostream>
#include <NTL/lzz_pX.h>
#include <NTL/ZZ.h>
#include "newinterp.h"
#include "GSinterp.h"

using namespace std;
using namespace NTL;

int main(int argc, char** argv) {
  if (argc != 4) {
    cout << "Usage: " << argv[0] << " which-tests degree-bound sparsity-bound"
         << endl
         << "\tdegree-bound is given as log_2 D"
         << "\twhich-tests indicates which algorithms to test among:" << endl
         << "\t\t0: Garg & Schost deterministic method" << endl
         << "\t\t1: Garg & Schost with probabilistic improvements" << endl
         << "\t\t2: Our probabilistic method, as presented in paper" << endl
         << "\t\t3: Our probabilistic method, with adaptive improvements"
         << endl
         << "\tby writing 1 for that bit." << endl;
    return 1;
  }

  unsigned long which = strtol(argv[1],NULL,2);
  long logD = atol(argv[2]);
  ZZ D;
  conv (D,1);
  D <<= logD;
  long T = atol(argv[3]);

  randseed();
  zz_p::init(65521);

  SparsePoly<zz_pX,zz_p> f;
  random(f,D,T);
  FakeUniModBB<zz_pX,zz_p> bb(f);

  double times[4];
  SparsePoly<zz_pX,zz_p> outs[4];
  bool pass;

  if (which & 1) {
    times[0] = -GetTime();
    gs_determ_interp (outs[0], bb, D, T);
    times[0] += GetTime();
    if (outs[0] != f) {
      cout << "G&S Deterministic Failed!" << endl;
    }
    cout << times[0] << " seconds for G&S Deterministic" << endl;
  }
  if (which & 2) {
    times[1] = -GetTime();
    gs_prob_interpB (outs[1], bb, D, T);
    times[1] += GetTime();
    if (outs[1] != f) {
      cout << "G&S Probabilistic Failed!" << endl;
    }
    cout << times[1] << " seconds for G&S Probabilistic" << endl;
  }
  if (which & 4) {
    times[2] = -GetTime();
    new_interpA (outs[2], bb, D, T);
    times[2] += GetTime();
    if (outs[2] != f) {
      cout << "Our new method version A failed!" << endl;
    }
    cout << times[2] << " seconds for new method A" << endl;
  }
  if (which & 8) {
    times[3] = -GetTime();
    pass = new_interpB (outs[3], bb, D, T);
    times[3] += GetTime();
    if ((!pass) || (outs[3] != f)) {
      cout << "Our new method version B failed!" << endl;
    }
    cout << times[3] << " seconds for new method B" << endl;
  }

  cout << "For LaTeX:" << endl;
  cout.precision(4);
  cout << logD << " & "
       << T;
  for (long i=0; i < 4; ++i) {
    cout << " & ";
    if (which & (1ULL<<i)) cout << times[i];
    else cout << "---";
  }
  cout << endl;

  return 0;
}
