/* sparsepoly.inc
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Some utilities for dealing with sparse polynomials in NTL.
 * (Note: for our purposes, all polynomials are univariate.)
 *
 * Include file (template implementations)
 */

#include <cassert>
#include <set>

NTL_OPEN_NNS

// Conversion from NTL poly to SparsePoly
template <typename Poly, typename Base>
void conv (SparsePoly<Poly,Base> &a, const Poly &b) {
  a.rep.clear();
  for (long i = 0; i <= deg(b); ++i) {
    if (!IsZero(coeff(b,i))) {
      a.rep.resize(a.rep.size()+1);
      a.rep.back().first = coeff(b,i);
      conv(a.rep.back().second, i);
    }
  }
}

/* Returns true iff f could be represented as a dense polynomial */
template <typename Poly, typename Base>
static inline bool smalldeg (const SparsePoly<Poly,Base> &f) {
  ZZ degf;
  deg(degf,f);
  return NumBits(degf) <= NTL_BITS_PER_LONG - 1;
}

/* Conversion from SparsePoly to NTL poly */
template <typename Poly, typename Base>
void conv (Poly &a, const SparsePoly<Poly,Base> &b) {
  assert (smalldeg(b));
  clear(a);

  for (typename SparsePoly<Poly,Base>::RepT::const_iterator
       iter = b.rep.begin();
       iter != b.rep.end();
       ++iter)
    SetCoeff (a, to_long(iter->second), iter->first);
}

// Random SparsePoly with deg < d and sparsity <= t
template <typename Poly, typename Base>
void random (SparsePoly<Poly,Base> &f, const ZZ& d, long t) {
  // Use STL set to build unique, sorted list of exponents
  std::set<ZZ> exset;
  for (long i=0; i<t; ++i)
    exset.insert(RandomBnd(d));

  f.rep.clear();
  for (std::set<ZZ>::iterator iter = exset.begin();
       iter != exset.end(); ++iter)
  {
    f.rep.resize(f.rep.size()+1);
    do { random (f.rep.back().first); }
    while (IsZero(f.rep.back().first));
    f.rep.back().second = *iter;
  }
}

// Output a SparsePoly
template <typename Poly, typename Base>
std::ostream& operator<< (std::ostream &out, const SparsePoly<Poly,Base> &f)
{
  out << "[ ";
  for (typename SparsePoly<Poly,Base>::RepT::const_iterator
       iter = f.rep.begin();
       iter != f.rep.end();
       ++iter)
    out << '(' << iter->first << ", " << iter->second << ") ";
  return out << ']';
}

NTL_CLOSE_NNS
