/* newinterp.h
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Black-box interpolation over a finite field using diversification.
 * See "Diversification improves interpolation", Giesbrecht & Roche, 2011
 *
 * Header file
 */

#ifndef NEWINTERP_H
#define NEWINTERP_H

#include <NTL/ZZ.h>
#include "blackbox.h"

NTL_OPEN_NNS

/* Tests whether the given polynomial is "diverse", i.e.
 * has all coefficients pairwise distinct.
 * The second template parameter is unfortunate but necessitated
 * because of NTL's design.
 */
template <typename Poly, typename Base>
bool is_diverse (const Poly& f);

/* Probabilistic interpolation method "A".
 * This is a mostly straightforward implementation of the method in
 * "Diversification improves interpolation", Giesbrecht & Roche, 2011,
 * made somewhat more adaptive by combining all "for" loops into one.
 * The size of the base field must be at least max(17,18*t^2*lnln t),
 * where t is the true sparsity of the unknown polynomial f.
 * The verification step is not included; the algorithm is Monte Carlo.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 */
template <typename BBT>
void new_interpA 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T);

/* Probabilistic interpolation method "B".
 * Improved method that only samples one "large enough" prime and then
 * looks for small good primes.
 * This Monte Carlo algorithm may return "false" on failure.
 * f: will hold the output
 * BBT: should be a subclass of UniModBB
 * bb: uni-modular black box for unknown polynomial f
 * D: upper bound on degree of f
 * T: upper bound on sparsity of f
 * Return: false if a known failure occurred, otherwise true
 */
template <typename BBT>
bool new_interpB 
  (SparsePoly< typename BBT::PolyT, typename BBT::BaseT >& f, 
   BBT& bb, ZZ D, long T);

NTL_CLOSE_NNS

#include "newinterp.inc"

#endif // NEWINTERP_H
