/* sparsepoly.h
 * Daniel S. Roche, January 2011
 * See COPYING.txt for permissions.
 *
 * Some utilities for dealing with sparse polynomials in NTL.
 * (Note: for our purposes, all polynomials are univariate.)
 *
 * Header file
 */

#ifndef SPARSEPOLY_H
#define SPARSEPOLY_H

#include <ostream>
#include <vector>
#include <NTL/ZZ.h>

NTL_OPEN_NNS

/* Compute the sparsity of an NTL polynomial */
template <typename Poly>
inline long sparsity (const Poly &f) {
  long toRet = 0;
  for (long i=0; i<=deg(f); ++i)
    if (!IsZero(f.rep[i])) ++toRet;
  return toRet;
}

/* A sparse polynomial is parameterized over the coefficient type "Base"
 * and is represented by an STL vector of STL pairs of
 * nonzero coefficient and exponent, sorted by increasing exponents. 
 * The exponent type is NTL::ZZ.
 *
 * The "Poly" type must be the NTL univariate polynomial type over "Base".
 *
 * The functionality is very limited, just to what is necessary for
 * our purposes here. The internal storage is exposed for further
 * functionality. I am imitating NTL's style of function calls even
 * though I don't really like it. The exception is the use of templates.
 */ 
template <typename Poly, typename Base>
class SparsePoly {
  public:
    typedef std::pair<Base, ZZ> TermT;
    typedef std::vector<TermT> RepT;
    RepT rep;
};

// Degree of a SparsePoly
template <typename Poly, typename Base>
inline void deg (ZZ &out, const SparsePoly<Poly,Base> &f) {
  if (f.rep.empty()) conv(out,-1);
  else out = f.rep.back().second;
}

// SparsePoly comparison
template <typename Poly, typename Base>
inline bool operator== 
  (const SparsePoly<Poly,Base> &a, const SparsePoly<Poly,Base> &b)
{
  return (a.rep == b.rep);
}

template <typename Poly, typename Base>
inline bool operator!= 
  (const SparsePoly<Poly,Base> &a, const SparsePoly<Poly,Base> &b)
  { return !(a == b); }

// Conversion from NTL poly to SparsePoly
template <typename Poly, typename Base>
void conv (SparsePoly<Poly,Base> &a, const Poly &b);

/* Conversion from SparsePoly to NTL poly
 * All the exponents must fit into long's of course.
 */
template <typename Poly, typename Base>
void conv (Poly &a, const SparsePoly<Poly,Base> &b);

// Random SparsePoly with deg < d and sparsity <= t
template <typename Poly, typename Base>
void random (SparsePoly<Poly,Base> &f, const ZZ& d, long t);

// Output a SparsePoly
template <typename Poly, typename Base>
std::ostream& operator<< (std::ostream &out, const SparsePoly<Poly,Base> &f);

NTL_CLOSE_NNS

#include "sparsepoly.inc"

#endif // SPARSEPOLY_H
